<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Settings;
use App\Models\Withdrawal;
use App\Models\UserWallet;
use App\Models\CryptoPrice;
use App\Models\Transaction;
use Illuminate\Support\Facades\Auth;
use App\Mail\NewNotification;
use Illuminate\Support\Facades\Mail;
use App\Mail\WithdrawalStatus;
use App\Traits\Coinpayment;
use App\Traits\TemplateTrait;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use App\Services\PaystackService;
use Illuminate\Support\Str;

class WithdrawalController extends Controller
{

    public function completewithdrawal(Request $request)
{
    try {
        $data = $request->validate([
            'currency_symbol'     => 'required|string',
            'amount'              => 'required|numeric|min:1',
            'wallet_address'      => 'nullable|string',
            'bank_account_number' => 'nullable|string',
            'bank_name'           => 'nullable|string',
            'bank_account_name'   => 'nullable|string',
        ]);

        $user = Auth::user();
        $settings = Settings::first();
        $symbol = $data['currency_symbol'];

        $wallet = UserWallet::where('user_id', $user->id)
            ->whereHas('currency', function($q) use ($symbol) {
                $q->where('symbol', $symbol);
            })->firstOrFail();

        $currency = $wallet->currency;

        $cryptoPrice = CryptoPrice::where('symbol', $symbol)->value('price');
        if (!$cryptoPrice) {
            return response()->json(['message' => 'Unable to fetch price for selected currency.'], 422);
        }

        $usdAmount = $data['amount'];
        $cryptoAmount = $usdAmount / $cryptoPrice;

        if ($wallet->balance < $cryptoAmount) {
            return response()->json(['message' => 'Insufficient balance.'], 422);
        }

        $ref = '#WD-' . strtoupper(Str::random(10));

        if ($symbol == 'USD') {
            if (empty($data['bank_account_number']) || empty($data['bank_name']) || empty($data['bank_account_name'])) {
                return response()->json(['message' => 'Complete bank details are required for USD withdrawal.'], 422);
            }
            $withdraw_to = $data['bank_name'] . ' - ' . $data['bank_account_number'];
            $cryptoAddress = null;
        } else {
            if (empty($data['wallet_address'])) {
                return response()->json(['message' => 'Wallet address is required for crypto withdrawal.'], 422);
            }
            $withdraw_to = $data['wallet_address'];
            $cryptoAddress = $data['wallet_address'];
        }

        $wallet->balance -= $cryptoAmount;
        $wallet->save();

        $type = strtoupper($symbol) === 'USD' ? 'Bank Withdrawal' : 'Crypto Withdrawal';

        $send = Transaction::create([
            'user_id'             => $user->id,
            'type'                => $type,
            'from'                => 'My ' . $currency->name . ' Wallet',
            'to'                  => $withdraw_to,
            'currency_id'         => $currency->id,
            'amount'              => $usdAmount,
            'crypto_address'      => $cryptoAddress,
            'bank_account_number' => $data['bank_account_number'] ?? null,
            'bank_name'           => $data['bank_name'] ?? null,
            'bank_account_name'   => $data['bank_account_name'] ?? null,
            'ref'                 => $ref,
            'hash'                => null,
            'symbol'              => $symbol,
            'payment_mode'        => ($symbol == 'USD') ? 'Bank Transfer' : 'Crypto',
            'status'              => 'Pending',
            'transaction_time'    => now(),
        ]);

        Mail::to($settings->contact_email)->send(new NewNotification(
            "A new withdrawal request of <b>{$usdAmount} {$symbol}</b> has been submitted by {$user->name}. Reference: {$ref}.",
            'New Withdrawal Request',
            'Admin'
        ));

        Mail::to($user->email)->send(new NewNotification(
            "Your withdrawal request of $<b>{$usdAmount}</b> has been received and is being processed. Reference: {$ref}.",
            'Withdrawal Request Received',
            $user->name
        ));

        return response()->json([
            'message'  => 'Withdrawal request submitted successfully!',
            'redirect' => route('status', $send->id)
        ]);
    } catch (\Exception $e) {
        Log::error('Withdrawal Processing Failed', ['error' => $e->getMessage()]);
        return response()->json(['message' => 'An error occurred while processing your withdrawal.'], 500);
    }
}

    
}
