<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\Currency;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\DepositStatus;
use App\Models\CryptoPrice;
use Illuminate\Support\Str;

class DepositController extends Controller
{
    public function newdeposit(Request $request)
    {
        try {
            $data = $request->validate([
                'currency_symbol' => 'required|string|exists:currencies,symbol',
                'amount'          => 'required|numeric|min:0.01',
            ]);

            session([
                'currency_symbol' => $data['currency_symbol'],
                'amount'          => $data['amount'],
            ]);

            return redirect()->route('payment');

        } catch (\Exception $e) {
            Log::error('New Deposit Error: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred while initiating deposit.'], 500);
        }
    }

    public function payment(Request $request)
    {
        try {
            $currencySymbol = session('currency_symbol');
            $amount         = session('amount');
            $currency = Currency::where('symbol', $currencySymbol)->first();

            if (! $currency || ! $currency->address) {
                return redirect()->back()->with('error', 'Currency address is unavailable.');
            }

            $paymentMode = $currency->name;
            $cryptoPrice = CryptoPrice::where('symbol', $currencySymbol)->first();
            $price = $cryptoPrice ? $cryptoPrice->price : null;

            return view('user.payment', [
                'amount'          => $amount,
                'payment_mode'    => $paymentMode,
                'wallet_address'  => $currency->address,
                'currency_symbol' => $currencySymbol,
                'currency_name'   => $currency->name,
                'crypto_price'    => $price,
                'title'           => 'Wallet Topup',
                'currency_id'     => $currency->id,
                'logo'            => $currency->logo,
            ]);

        } catch (\Exception $e) {
            Log::error('Payment Page Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load payment page.');
        }
    }

    public function savedeposit(Request $request)
    {
        try {
            $request->validate([
                'hash' => 'nullable|string',
            ]);

            $currencySymbol = session('currency_symbol');
            $amount         = session('amount');
            $paymentMode    = $request->payment_mode;
            $currency_id    = $request->currency_id;
            $to             = 'My ' . $paymentMode . ' Wallet';
            $type           = $currencySymbol == 'USD' ? 'Bank Deposit' : 'Crypto Deposit';
            $ref            = '#HLP-' . Str::upper(Str::random(7));

            $deposit = Transaction::create([
                'amount'       => $amount,
                'payment_mode' => $paymentMode,
                'status'       => 'Pending',
                'hash'         => $request->hash,
                'user_id'      => Auth::id(),
                'symbol'       => $currencySymbol,
                'type'         => $type, 
                'currency_id'  => $currency_id,
                'ref'          => $ref,
                'from'         => 'External Wallet',
                'to'           => $to
            ]);

            try {
                Mail::to(Auth::user()->email)
                    ->send(new DepositStatus($deposit, Auth::user(), 'Deposit Received', false));
            } catch (\Exception $e) {
                Log::warning('Failed to send deposit email: ' . $e->getMessage());
            }

            session()->forget(['currency_symbol', 'amount', 'payment_mode']);

            return redirect()->route('status', $deposit->id)->with('message', 'Deposit pending, Awaiting Network confirmation.');

        } catch (\Exception $e) {
            Log::error('Save Deposit Error: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred while saving your deposit.'], 500);
        }
    }
}
