<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CryptoPrice;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class CryptoPriceController extends Controller
{
    public function updatePrices()
{
    try {
        $response = Http::get('https://api.coingecko.com/api/v3/coins/markets', [
            'vs_currency' => 'usd',
            'order' => 'market_cap_desc',
            'per_page' => 50,
            'page' => 1,
            'sparkline' => false,
            'price_change_percentage' => '24h'
        ]);

        if ($response->failed()) {
            Log::error('Failed to fetch crypto data from CoinGecko.');
            return response()->json(['message' => 'Failed to fetch data from CoinGecko'], 500);
        }

        $cryptos = $response->json();

        foreach ($cryptos as $coin) {
            if ($coin['id'] === 'tether') {
                // USDT for TRC20 and ERC20
                CryptoPrice::updateOrCreate(
                    ['crypto_name' => $coin['id'], 'network' => 'TRC20'],
                    [
                        'symbol' => 'USDT.TRC20',
                        'price' => $coin['current_price'],
                        'usd_price' => $coin['current_price'],
                        'percent_change_24h' => $coin['price_change_percentage_24h_in_currency'] ?? null,
                        'logo' => $coin['image'] // store external URL
                    ]
                );

                CryptoPrice::updateOrCreate(
                    ['crypto_name' => $coin['id'], 'network' => 'ERC20'],
                    [
                        'symbol' => 'USDT.ERC20',
                        'price' => $coin['current_price'],
                        'usd_price' => $coin['current_price'],
                        'percent_change_24h' => $coin['price_change_percentage_24h_in_currency'] ?? null,
                        'logo' => $coin['image']
                    ]
                );
            } else {
                CryptoPrice::updateOrCreate(
                    ['symbol' => strtoupper($coin['symbol'])],
                    [
                        'crypto_name' => $coin['id'],
                        'network' => 'mainnet',
                        'price' => $coin['current_price'],
                        'usd_price' => $coin['current_price'],
                        'percent_change_24h' => $coin['price_change_percentage_24h_in_currency'] ?? null,
                        'logo' => $coin['image'] // just use the URL
                    ]
                );
            }
        }

        return response()->json(['message' => 'Crypto prices updated successfully.']);

    } catch (\Exception $e) {
        Log::error('Error updating crypto prices: ' . $e->getMessage());
        return response()->json(['message' => 'An error occurred.'], 500);
    }
}


    private function downloadLogo($url, $filename)
    {
        try {
            $contents = file_get_contents($url);
            $ext = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
            $filePath = "crypto_logos/{$filename}.{$ext}";

            Storage::disk('public')->put($filePath, $contents);

            return $filePath;
        } catch (\Exception $e) {
            Log::warning("Failed to download logo for {$filename}: " . $e->getMessage());
            return null;
        }
    }
}
