<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Settings;
use App\Models\Deposit;
use App\Models\Transaction;
use App\Models\Withdrawal;
use App\Models\UserWallet;
use App\Models\CardTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\CryptoPrice;
use App\Models\Contact;
use App\Models\Card;
use App\Models\Bill;
use App\Helpers\WalletHelper;
use Barryvdh\DomPDF\Facade\Pdf;
use PragmaRX\Google2FAQRCode\Google2FA;




class ViewsController extends Controller
{
   
    
    public function dashboard(Request $request)
{
    try {
        $settings = Settings::find(1);
        $user = Auth::user();

        $userWallets = UserWallet::with('currency')->where('user_id', $user->id)->get();
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $sym = $wallet->currency->symbol;
            $price = $cryptoPrices[$sym]->price ?? 0;
            $usd = $wallet->balance * $price;
            $totalBalanceInUSD += $usd;
            return [
                'currency'         => $wallet->currency->name,
                'symbol'           => $sym,
                'logo'             => $wallet->currency->logo,
                'balance'          => $wallet->balance,
                'balance_in_usd'   => $usd,
                'percent_change_24h'=> $cryptoPrices[$sym]->percent_change_24h ?? 0,
            ];
        });

        $recentTransactions = Transaction::with('currency:id,symbol,logo,name')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get()
            ->map(function($tx) use ($cryptoPrices) {
                if ($tx->currency) {
                    $sym = $tx->currency->symbol;
                    $price = $cryptoPrices[$sym]->price ?? 0;
                    $tx->amount_in_usd = $tx->amount * $price;
                } else {
                    $tx->amount_in_usd = $tx->amount;
                }
                return $tx;
            });

        if ($user->status !== 'active') {
            $request->session()->flush();
            return redirect()->route('dashboard');
        }

        return view('user.dashboard', [
            'title'               => 'Account Dashboard',
            'userWalletsInUSD'    => $userWalletsInUSD,
            'totalBalanceInUSD'   => $totalBalanceInUSD,
            'cryptoPrices'        => $cryptoPrices,
            'recentTransactions'  => $recentTransactions,
        ]);
    } catch (\Exception $e) {
        Log::error('Dashboard Load Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load dashboard.');
    }
}

public function transactions()
{
    try {
        $user = Auth::user();
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        $recentTransactions = Transaction::with('currency:id,symbol,logo,name')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $recentTransactions->getCollection()->transform(function($tx) use ($cryptoPrices) {
            $sym = $tx->currency->symbol ?? 'USD';
            $price = $cryptoPrices[$sym]->price ?? 0;
            $tx->amount_in_usd = $tx->amount * $price;
            return $tx;
        });

        $first = $recentTransactions->first();
        $defaultLogo = $first && $first->currency ? $first->currency->logo : 'default-logo.png';
        $defaultSymbol = $first && $first->currency ? $first->currency->symbol : 'USD';

        return view('user.transactions', [
            'title'        => 'Account Transactions History',
            'transactions' => $recentTransactions,
            'logo'         => $defaultLogo,
            'symbol'       => $defaultSymbol,
            'cryptoPrices' => $cryptoPrices,
        ]);
    } catch (\Exception $e) {
        Log::error('Transaction History Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load transactions.');
    }
}

public function cardTransactions()
{
    try {
        $user = Auth::user();
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        $recentTransactions = CardTransaction::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $recentTransactions->getCollection()->transform(function($tx) use ($cryptoPrices) {
            $sym = $tx->currency->symbol ?? 'USD';
            $price = $cryptoPrices[$sym]->price ?? 0;
            $tx->amount_in_usd = $tx->amount * $price;
            return $tx;
        });

        $first = $recentTransactions->first();
        $defaultLogo = $first && $first->currency ? $first->currency->logo : 'default-logo.png';
        $defaultSymbol = $first && $first->currency ? $first->currency->symbol : 'USD';

        return view('user.card-transactions', [
            'title'        => 'Card Transactions History',
            'transactions' => $recentTransactions,
            'logo'         => $defaultLogo,
            'symbol'       => $defaultSymbol,
            'cryptoPrices' => $cryptoPrices,
        ]);
    } catch (\Exception $e) {
        Log::error('Card Transaction History Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load card transactions.');
    }
}

public function deposits()
{
    try {
        $user = Auth::user();
        $userWallets = UserWallet::where('user_id', $user->id)->with('currency')->get();
        $cryptoPrices = CryptoPrice::all()->pluck('price', 'symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $currencySymbol = $wallet->currency->symbol;
            $currentPriceInUSD = $cryptoPrices[$currencySymbol] ?? 0;
            $balanceInUSD = $wallet->balance * $currentPriceInUSD;
            $totalBalanceInUSD += $balanceInUSD;
            return [
                'currency' => $wallet->currency->name,
                'symbol' => $wallet->currency->symbol,
                'balance' => $wallet->balance,
                'balance_in_usd' => $balanceInUSD,
                'name' => $wallet->currency->name,
                'logo' => $wallet->currency->logo,
            ];
        });

        $total_deposited = DB::table('deposits')->where('user', $user->id);

        return view("user.deposits", [
            'title' => 'Receive funds',
            'userWalletsInUSD' => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'deposits' => Deposit::where(['user' => $user->id]),
            'deposited' => $total_deposited,
        ]);
    } catch (\Exception $e) {
        Log::error('Deposits Page Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load deposits.');
    }
}

public function withdrawals()
{
    try {
        $user = Auth::user();
        $userWallets = UserWallet::where('user_id', $user->id)->with('currency')->get();
        $cryptoPrices = CryptoPrice::all()->pluck('price', 'symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $currencySymbol = $wallet->currency->symbol;
            $currentPriceInUSD = $cryptoPrices[$currencySymbol] ?? 0;
            $balanceInUSD = $wallet->balance * $currentPriceInUSD;
            $totalBalanceInUSD += $balanceInUSD;
            return [
                'currency' => $wallet->currency->name,
                'symbol' => $wallet->currency->symbol,
                'balance' => $wallet->balance,
                'balance_in_usd' => $balanceInUSD,
                'name' => $wallet->currency->name,
                'logo' => $wallet->currency->logo,
            ];
        });

        $total_deposited = DB::table('deposits')->where('user', $user->id);

        return view("user.withdrawals", [
            'title' => 'Receive funds',
            'userWalletsInUSD' => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'deposits' => Deposit::where(['user' => $user->id]),
            'deposited' => $total_deposited,
        ]);
    } catch (\Exception $e) {
        Log::error('Withdrawals Page Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load withdrawals.');
    }
}

public function coinDetails($symbol)
{
    try {
        $user = Auth::user();
        $wallet = UserWallet::where('user_id', $user->id)
            ->whereHas('currency', function ($query) use ($symbol) {
                $query->where('symbol', $symbol);
            })->with('currency')->first();

        $cryptoPrice = CryptoPrice::where('symbol', $symbol)->first();
        if (!$cryptoPrice) {
            return redirect()->back()->with('message', 'Unable to fetch price for the selected currency.');
        }

        $balanceInUSD = $wallet->balance * $cryptoPrice->price;

        $deposits = Deposit::where('user_wallet_id', $wallet->id)
            ->where('payment_mode', $symbol)
            ->orderBy('created_at', 'desc')
            ->get();

        $deposits->each(function ($deposit) use ($cryptoPrice) {
            $deposit->usd_amount = $deposit->amount * $cryptoPrice->price;
        });

        $withdrawals = Withdrawal::where('user', $user->id)
            ->where('payment_mode', $symbol)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('user.coin-details', [
            'wallet' => $wallet,
            'deposits' => $deposits,
            'withdrawals' => $withdrawals,
            'symbol' => $symbol,
            'balanceInUSD' => $balanceInUSD,
            'title' => 'Coin Details'
        ]);
    } catch (\Exception $e) {
        Log::error('Coin Details Load Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load coin details.');
    }
}


public function details($symbol, $id)
{
    try {
        $user = Auth::user();

        $wallet = UserWallet::where('user_id', $user->id)
            ->whereHas('currency', function ($query) use ($symbol) {
                $query->where('symbol', $symbol);
            })->with('currency')->first();

        if (!$wallet) {
            return redirect()->back()->with('message', 'Wallet not found for this currency.');
        }

        $cryptoPrice = CryptoPrice::where('symbol', $symbol)->first();
        if (!$cryptoPrice) {
            return redirect()->back()->with('message', 'Unable to fetch price for the selected currency.');
        }

        $balanceInUSD = $wallet->balance * $cryptoPrice->price;

        $transaction = Transaction::where('user_wallet_id', $wallet->id)
            ->where('id', $id)
            ->where('payment_mode', $symbol)
            ->first();

        if (!$transaction) {
            return redirect()->back()->with('message', 'Transaction not found.');
        }

        $amountInUsd = $transaction->amount * $cryptoPrice->price;

        return view('user.details', [
            'wallet' => $wallet,
            't' => $transaction,
            'symbol' => $symbol,
            'balanceInUSD' => $balanceInUSD,
            'title' => 'Transaction Details',
            'amountInUsd' => $amountInUsd,
        ]);
    } catch (\Exception $e) {
        Log::error('Transaction Details Error', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load transaction details.');
    }
}

public function showStatus($id)
{
    try {
        $transaction = Transaction::with('currency')
            ->where('id', $id)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        return view('user.status', [
            'transaction' => $transaction,
            'title' => 'Transaction Details',
        ]);
    } catch (\Exception $e) {
        Log::error('Show Status Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Transaction not found.');
    }
}

public function twofa()
{
    return view("profile.show", ['title' => 'Advance Security Settings']);
}

public function verifyaccount()
{
    if (Auth::user()->account_verify == 'Verified') {
        abort(404, 'You do not have permission to access this page');
    }
    return view("user.verify", ['title' => 'Verify your Account']);
}

public function verificationForm()
{
    if (Auth::user()->account_verify == 'Verified') {
        abort(404, 'You do not have permission to access this page');
    }
    return view("user.verification", ['title' => 'KYC Application']);
}

public function insights()
{
    try {
        $user = Auth::user();
        $transactions = Transaction::where('user_id', $user->id);
        $totalDeposits = (clone $transactions)->where('type', 'LIKE', '%Deposit%')->sum('amount');
        $totalWithdrawals = (clone $transactions)->where('type', 'LIKE', '%Withdraw%')->sum('amount');

        $deposits = DB::table('transactions')
            ->select(
                DB::raw('MONTH(created_at) as month_number'),
                DB::raw('DATE_FORMAT(created_at, "%b") as month_name'),
                DB::raw('SUM(amount) as total')
            )
            ->where('user_id', $user->id)
            ->where('type', 'LIKE', '%Deposit%')
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy(DB::raw('MONTH(created_at)'), DB::raw('DATE_FORMAT(created_at, "%b")'))
            ->orderBy('month_number')
            ->pluck('total', 'month_name');

        $months = [];
        $totals = [];
        for ($i = 5; $i >= 0; $i--) {
            $monthName = now()->subMonths($i)->format('M');
            $months[] = $monthName;
            $totals[] = $deposits[$monthName] ?? 0;
        }

        $chartData = ['months' => $months, 'deposits' => $totals];

        $counts = [
            'income'      => (clone $transactions)->where('type', 'LIKE', '%Deposit%')->count(),
            'sent'        => (clone $transactions)->where('type', 'LIKE', '%Send%')->count(),
            'request'     => (clone $transactions)->where('type', 'LIKE', '%Request%')->count(),
            'topup'       => (clone $transactions)->where('type', 'LIKE', '%Top Up%')->count(),
            'withdraw'    => (clone $transactions)->where('type', 'LIKE', '%Withdraw%')->count(),
            'transaction' => (clone $transactions)->count(),
        ];

        return view('user.insights', [
            'title' => 'Statistics',
            'settings' => Settings::first(),
            'chartData' => $chartData,
            'totalDeposits' => $totalDeposits,
            'totalWithdrawals' => $totalWithdrawals,
            'counts' => $counts,
        ]);
    } catch (\Exception $e) {
        Log::error('Insights Load Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load insights.');
    }
}

public function PersonalProfile()
{
    $userinfo = Auth::user();
    return view("user.personal-profile", ['userinfo' => $userinfo, 'title' => 'Profile']);
}

public function PasswordUpdate()
{
    $userinfo = Auth::user();
    return view("user.password-update", ['userinfo' => $userinfo, 'title' => 'Update Password']);
}

public function WithdrawalInformation()
{
    $userinfo = Auth::user();
    return view("user.withdrawalinformation", ['userinfo' => $userinfo, 'title' => 'Withdrawal Information']);
}

public function UpdateBank()
{
    $userinfo = Auth::user();
    $banks = DB::table('banks')->orderBy('name')->get();
    return view('user.updatebank', ['userinfo' => $userinfo, 'banks' => $banks, 'title' => 'Update Bank Information']);
}

public function Security(Request $request)
{
    $userinfo = Auth::user();
    return view("user.security", ['userinfo' => $userinfo, 'title' => 'Security Settings']);
}

public function profile()
{
    $userinfo = Auth::user();
    return view("user.profile", ['userinfo' => $userinfo, 'title' => 'Profile']);
}

public function aboutUs()
{
    return view("user.about-us", ['title' => 'About Us']);
}

public function policy()
{
    return view("user.policy", ['title' => 'Privacy Policy']);
}

public function accountdetails()
{
    return view("user.updateacct", ['title' => 'Update account details']);
}

public function notifications()
{
    $userinfo = Auth::user();
    return view("user.notifications", ['title' => 'Notifications', 'userinfo' => $userinfo]);
}

public function support()
{
    return view("user.support", ['title' => 'Support']);
}

public function contacts()
{
    $contacts = Contact::where('user_id', Auth::id())->orderBy('name')->get();
    return view('user.contacts', ['title' => 'My Contacts', 'contacts' => $contacts]);
}

public function viewcontacts()
{
    $contacts = Contact::where('user_id', Auth::id())->orderBy('name')->get();
    return view('user.viewcontacts', ['title' => 'My Contacts', 'contacts' => $contacts]);
}

public function editcontacts($id)
{
    $contact = Contact::where('user_id', Auth::id())->where('id', $id)->firstOrFail();
    return view('user.editcontacts', ['title' => 'Edit Contacts', 'contact' => $contact]);
}

public function addContact()
{
    return view('user.add-contact', ['title' => 'Add Contacts']);
}

public function contactdetails($id)
{
    $contact = Contact::where('user_id', Auth::id())->where('id', $id)->firstOrFail();
    return view('user.contactdetails', ['title' => 'Edit Contacts', 'contact' => $contact]);
}

public function send()
{
    return view('user.send', ['title' => 'Send Money']);
}


public function sendToContact($id)
{
    try {
        $settings = Settings::find(1);
        $user = Auth::user();
        $userWallets = UserWallet::where('user_id', $user->id)->with('currency')->get();
        $cryptoPrices = CryptoPrice::all()->pluck('price', 'symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $currencySymbol = $wallet->currency->symbol;
            $currentPriceInUSD = $cryptoPrices[$currencySymbol] ?? 0;
            $balanceInUSD = $wallet->balance * $currentPriceInUSD;
            $totalBalanceInUSD += $balanceInUSD;
            return [
                'currency' => $wallet->currency->name,
                'symbol' => $wallet->currency->symbol,
                'balance' => $wallet->balance,
                'balance_in_usd' => $balanceInUSD,
                'name' => $wallet->currency->name,
                'logo' => $wallet->currency->logo,
            ];
        });

        $contact = Contact::where('user_id', $user->id)->where('id', $id)->firstOrFail();
        $selectedWallet = $userWalletsInUSD->first();
        $rate = ($contact->type == 'bank') ? DB::table('rates')->where('base_currency', 'USD')->where('target_currency', $contact->currency)->value('rate') ?? 0 : 0;

        return view('user.send-contact', [
            'title' => 'Send Money to ' . $contact->name,
            'contact' => $contact,
            'settings' => $settings,
            'userWalletsInUSD' => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'rate' => $rate,
            'selectedWallet' => $selectedWallet
        ]);
    } catch (\Exception $e) {
        Log::error('Send to Contact Failed', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load contact send page.');
    }
}

public function sendToUsername()
{
    $settings = Settings::find(1);
    $user = Auth::user();
    $users = User::where('id', '!=', $user->id)->get();
    $userWallets = UserWallet::where('user_id', $user->id)->with('currency')->get();
    $cryptoPrices = CryptoPrice::all()->pluck('price', 'symbol');

    $totalBalanceInUSD = 0;
    $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
        $currencySymbol = $wallet->currency->symbol;
        $currentPriceInUSD = $cryptoPrices[$currencySymbol] ?? 0;
        $balanceInUSD = $wallet->balance * $currentPriceInUSD;
        $totalBalanceInUSD += $balanceInUSD;
        return [
            'currency' => $wallet->currency->name,
            'symbol' => $wallet->currency->symbol,
            'balance' => $wallet->balance,
            'balance_in_usd' => $balanceInUSD,
            'name' => $wallet->currency->name,
            'logo' => $wallet->currency->logo,
        ];
    });

    return view('user.send-username', [
        'title' => 'Send Money to @tag',
        'settings' => $settings,
        'users' => $users,
        'userWalletsInUSD' => $userWalletsInUSD,
        'totalBalanceInUSD' => $totalBalanceInUSD
    ]);
}

public function cards()
{
    try {
        $user = Auth::user();
        $userWallets = UserWallet::with('currency')->where('user_id', $user->id)->get();
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $sym = $wallet->currency->symbol;
            $price = $cryptoPrices[$sym]->price ?? 0;
            $usd = $wallet->balance * $price;
            $totalBalanceInUSD += $usd;
            return [
                'balance' => $wallet->balance,
                'balance_in_usd' => $usd,
                'percent_change_24h' => $cryptoPrices[$sym]->percent_change_24h ?? 0,
            ];
        });

        $recentTransactions = CardTransaction::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        $cards = Card::where('user_id', $user->id)->get();

        return view('user.cards', [
            'title' => 'Cards',
            'userWalletsInUSD' => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'cryptoPrices' => $cryptoPrices,
            'recentTransactions' => $recentTransactions,
            'cards' => $cards,
        ]);
    } catch (\Exception $e) {
        Log::error('Cards Page Error', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load cards page.');
    }
}

public function payBills()
{
    try {
        $user = Auth::user();

        // Fetch user wallets with currency relationship
        $userWallets = UserWallet::with('currency')
            ->where('user_id', $user->id)
            ->get();

        // Fetch crypto prices indexed by symbol for quick access
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        // Calculate wallet balances in USD
        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $symbol = $wallet->currency->symbol;
            $pricePerUnit = $cryptoPrices[$symbol]->price ?? 0;
            $usdValue = $wallet->balance * $pricePerUnit;

            $totalBalanceInUSD += $usdValue;

            return [
                'currency'        => $wallet->currency->name,
                'symbol'          => $symbol,
                'balance'         => $wallet->balance,
                'balance_in_usd'  => $usdValue,
                'name'            => $wallet->currency->name,
                'logo'            => $wallet->currency->logo,
            ];
        });

        // Fetch user's cards
        $cards = Card::where('user_id', $user->id)->get();

        // Fetch all available bills
        $bills = Bill::all();

        return view('user.pay', [
            'title'             => 'Pay Bills',
            'userWalletsInUSD'  => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'cryptoPrices'      => $cryptoPrices,
            'cards'             => $cards,
            'bills'             => $bills,
        ]);
    } catch (\Exception $e) {
        Log::error('Pay Bills Page Error', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load pay bills page.');
    }
}


public function buyCrypto()
{
    try {
        $user = Auth::user();
        $userWallets = UserWallet::with('currency')->where('user_id', $user->id)->get();
        $cryptoPrices = CryptoPrice::all()->keyBy('symbol');

        $totalBalanceInUSD = 0;
        $userWalletsInUSD = $userWallets->map(function($wallet) use ($cryptoPrices, &$totalBalanceInUSD) {
            $sym = $wallet->currency->symbol;
            $price = $cryptoPrices[$sym]->price ?? 0;
            $usd = $wallet->balance * $price;
            $totalBalanceInUSD += $usd;
            return [
                'currency' => $wallet->currency->name,
                'symbol' => $wallet->currency->symbol,
                'balance' => $wallet->balance,
                'balance_in_usd' => $usd,
                'name' => $wallet->currency->name,
                'logo' => $wallet->currency->logo,
            ];
        });

        $cards = Card::where('user_id', $user->id)->get();
        $bills = Bill::all();
        $cryptos = CryptoPrice::orderBy('id')->take(30)->get(['crypto_name', 'symbol', 'logo']);

        return view('user.buy-crypto', [
            'title' => 'Pay Bills',
            'userWalletsInUSD' => $userWalletsInUSD,
            'totalBalanceInUSD' => $totalBalanceInUSD,
            'cryptoPrices' => $cryptoPrices,
            'cards' => $cards,
            'bills' => $bills,
            'cryptos' => $cryptos,
        ]);
    } catch (\Exception $e) {
        Log::error('Buy Crypto Page Error', ['error' => $e->getMessage()]);
        return redirect()->back()->with('error', 'Unable to load buy crypto page.');
    }
}


public function enableGoogle2FA()
{
    $google2fa = new Google2FA();

    $user = Auth::user();

    // Generate a new secret
    $secretKey = $google2fa->generateSecretKey();

    // Save to user
    $user->two_factor_secret = encrypt($secretKey);
    $user->save();

    // Generate QR code URL
    $QR_Image = $google2fa->getQRCodeInline(
        config('app.name'),
        $user->email,
        $secretKey
    );

    return view('user.2fa_setup', [
        'qrCode' => $QR_Image,
        'secret' => $secretKey
    ]);
}


 }