<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Auth\LoginController;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Settings;
use App\Models\Deposit;
use App\Models\Transaction;
use App\Models\Bill;
use App\Models\Card;
use App\Models\Contact;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\CryptoPrice;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use PragmaRX\Google2FAQRCode\Google2FA;
use App\Models\UserWallet;
use App\Models\Kyc;
use App\Models\CardTransaction;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use \App\Mail\ContactTransferSuccess;
use App\Mail\NewNotification;




class UserController extends Controller
{

    public function PasswordUpdate(Request $request)
    {
        try {
            $user = Auth::user();

            $validator = Validator::make($request->all(), [
                'current_password' => 'required|string',
                'new_password' => [
                    'required', 'string', 'min:8',
                    'regex:/[A-Z]/',
                    'regex:/[0-9]/',
                    'regex:/[!@#$%^&*(),.?":{}|<>[\]]/',
                    'confirmed'
                ],
                'new_password_confirmation' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
            }

            if (!Hash::check($request->current_password, $user->password)) {
                return response()->json(['success' => false, 'message' => 'The current password is incorrect.'], 401);
            }

            $user->password = Hash::make($request->new_password);
            $user->save();

            Auth::login($user);

            return response()->json(['success' => true, 'message' => 'Password updated successfully.']);
        } catch (\Exception $e) {
            Log::error('Password Update Error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'Failed to update password.'], 500);
        }
    }

    public function toggleFavorite(Contact $contact)
    {
        if ($contact->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $contact->liked = $contact->liked === 'yes' ? 'no' : 'yes';
        $contact->save();

        return response()->json(['liked' => $contact->liked]);
    }

    public function addContact(Request $request)
    {
        try {
            $data = $request->validate([
                'name' => 'required|string|max:100',
                'type' => 'required|string',
                'country' => 'required|string',
                'currency' => 'required|string',
                'bank_name' => 'nullable|required_if:type,bank|string|max:100',
                'bank_account_number' => 'nullable|required_if:type,bank|string|max:50',
                'routing_number' => 'nullable|required_if:type,bank|string|max:9',
                'account_type' => 'nullable|required_if:type,bank|in:checking,savings',
                'wallet_address' => 'nullable|required_unless:type,bank|string|max:255',
            ]);

            $data['user_id'] = Auth::id();
            $data['liked'] = 'no';

            Contact::create($data);

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Add Contact Error: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to add contact.'], 500);
        }
    }

    public function sendToContact(Request $request, $id)
    {
        try {
            $user = Auth::user();
            $contact = Contact::where('user_id', $user->id)->where('id', $id)->firstOrFail();

            $data = $request->validate([
                'currency_symbol' => 'required|string',
                'currency_name' => 'required|string',
                'amount' => 'required|numeric|min:1',
            ]);

            $cryptoPriceRecord = CryptoPrice::where('symbol', $data['currency_symbol'])->first();
            if (!$cryptoPriceRecord) {
                return response()->json(['message' => 'Unable to fetch crypto price.'], 422);
            }

            $cryptoAmount = $data['amount'] / $cryptoPriceRecord->price;

            $wallet = UserWallet::where('user_id', $user->id)
                        ->whereHas('currency', function($q) use ($data) {
                            $q->where('symbol', $data['currency_symbol']);
                        })->firstOrFail();

            if ($wallet->balance < $cryptoAmount) {
                return response()->json(['message' => 'Insufficient wallet balance.'], 422);
            }

            $wallet->balance -= $cryptoAmount;
            $wallet->save();

            $ref = '#TRF-' . Str::upper(Str::random(7));

            $send = Transaction::create([
                'user_id' => $user->id,
                'type' => 'Send to Contact',
                'currency_id' => $wallet->currency_id,
                'amount' => $data['amount'],
                'status' => 'Completed',
                'payment_mode' => $contact->type,
                'from' => 'My ' . $wallet->currency->name . ' Wallet',
                'to' => $contact->name,
                'symbol' => $data['currency_symbol'],
                'bank_name' => $contact->bank_name,
                'bank_account_number' => $contact->bank_account_number,
                'ref' => $ref
            ]);

            try {
                Mail::to($user->email)->send(new ContactTransferSuccess($user, $contact, $data['amount'], $contact->currency));
            } catch (\Exception $e) {
                Log::warning('Transfer email failed: ' . $e->getMessage());
            }

            return response()->json(['message' => "Transfer successful!", 'redirect' => route('status', $send->id)]);
        } catch (\Exception $e) {
            Log::error('Send to Contact Error: ' . $e->getMessage());
            return response()->json(['message' => 'Transfer failed.'], 500);
        }
    }

    public function editContact($id)
    {
        $contact = Contact::findOrFail($id);
        return view('contacts.edit', ['contact' => $contact, 'title' => 'Edit Contact']);
    }

    public function update(Request $request, $id)
    {
        try {
            $contact = Contact::findOrFail($id);

            $data = $request->validate([
                'name' => 'required|string|max:255',
                'country' => 'required|string',
                'currency' => 'required|string',
                'type' => 'required|string|in:bank,crypto,paypal,zelle,cashapp',
                'bank_name' => 'nullable|string',
                'bank_account_number' => 'nullable|string',
                'routing_number' => 'nullable|string',
                'account_type' => 'nullable|string|in:checking,savings',
                'wallet_address' => 'nullable|string',
            ]);

            $contact->update($data);

            return response()->json(['message' => 'Contact updated successfully.']);
        } catch (\Exception $e) {
            Log::error('Update Contact Error: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to update contact.'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $contact = Contact::findOrFail($id);
            $contact->delete();
            return response()->json(['message' => 'Contact deleted successfully.']);
        } catch (\Exception $e) {
            Log::error('Delete Contact Error: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to delete contact.'], 500);
        }
    }

    public function fetchUsernameDetails(Request $request)
    {
        try {
            $username = ltrim($request->username, '@');
            $user = User::where('username', $username)->first();

            return response()->json(['name' => $user ? trim($user->first_name . ' ' . $user->last_name) : null]);
        } catch (\Exception $e) {
            Log::error('Fetch Username Details Error: ' . $e->getMessage());
            return response()->json(['name' => null]);
        }
    }


    public function processSendToUsername(Request $request)
    {
        try {
            $data = $request->validate([
                'username'        => 'required|string',
                'currency_symbol' => 'required|string',
                'currency_name'   => 'required|string',
                'amount'          => 'required|numeric|min:1',
            ]);
    
            $sender = Auth::user();
            $receiverUsername = ltrim($data['username'], '@');
    
            $receiver = User::where('username', $receiverUsername)->first();
            if (!$receiver) {
                return response()->json(['message' => 'Recipient not found.'], 404);
            }
    
            $wallet = UserWallet::where('user_id', $sender->id)
                        ->whereHas('currency', function($q) use ($data) {
                            $q->where('symbol', $data['currency_symbol']);
                        })->firstOrFail();
    
            $cryptoPrice = CryptoPrice::where('symbol', $data['currency_symbol'])->value('price');
            if (!$cryptoPrice || $cryptoPrice <= 0) {
                return response()->json(['message' => 'Crypto price unavailable.'], 422);
            }
    
            $usdAmount = $data['amount'];
            $cryptoAmount = $usdAmount / $cryptoPrice;
            $feeInCrypto = $cryptoAmount * 0.015;
            $amountToCredit = $cryptoAmount - $feeInCrypto;
    
            if ($wallet->balance < $cryptoAmount) {
                return response()->json(['message' => 'Insufficient crypto balance.'], 422);
            }
    
            $wallet->balance -= $cryptoAmount;
            $wallet->save();
    
            $receiverWallet = UserWallet::where('user_id', $receiver->id)
                                ->where('currency_id', $wallet->currency_id)
                                ->first();
    
            if ($receiverWallet) {
                $receiverWallet->balance += $amountToCredit;
                $receiverWallet->save();
            }
    
            $currencyName = $wallet->currency->name;
            $ref = '#TRF-' . strtoupper(Str::random(10));
    
            $send = Transaction::create([
                'user_id'      => $sender->id,
                'type'         => 'Internal Transfer',
                'currency_id'  => $wallet->currency_id,
                'amount'       => $usdAmount,
                'status'       => 'completed',
                'payment_mode' => $currencyName,
                'from'         => 'My ' . $currencyName . ' Wallet',
                'to'           => '@' . $receiverUsername,
                'symbol'       => $data['currency_symbol'],
                'ref'          => $ref,
            ]);
    
            Transaction::create([
                'user_id'      => $receiver->id,
                'type'         => 'Internal Transfer',
                'currency_id'  => $wallet->currency_id,
                'amount'       => $amountToCredit * $cryptoPrice,
                'status'       => 'Completed',
                'payment_mode' => $currencyName,
                'from'         => '@' . $sender->username,
                'to'           => 'My ' . $currencyName . ' Wallet',
                'symbol'       => $data['currency_symbol'],
                'ref'          => $ref,
            ]);
    
            $senderBody = "You have successfully sent $" . number_format($usdAmount, 2) . " to @{$receiverUsername}. Reference: {$ref}.";
            $receiverBody = "You have received $" . number_format($amountToCredit * $cryptoPrice, 2) . " from @{$sender->username}. Reference: {$ref}.";
    
            Mail::to($sender->email)->send(new NewNotification($senderBody, 'Transfer Successful', $sender->first_name, route('status', $send->id)));
            Mail::to($receiver->email)->send(new NewNotification($receiverBody, 'You Received Funds!', $receiver->first_name, route('status', $send->id)));
    
            return response()->json([
                'message'  => 'Transfer successful! $' . number_format($usdAmount, 2) . ' sent to @' . $receiverUsername . '.',
                'redirect' => route('status', $send->id)
            ]);
        } catch (\Exception $e) {
            Log::error('ProcessSendToUsername Error: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred while processing transfer.'], 500);
        }
    }
    
    public function createCard(Request $request)
    {
        try {
            $user = Auth::user();
            $cardType = $request->card_type;
    
            $cardRules = [
                'visa_card' => ['limit' => 2000, 'required_tx' => 500],
                'master_card' => ['limit' => 20000, 'required_tx' => 2000],
                'gold_card' => ['limit' => 50000, 'required_tx' => 10000],
            ];
    
            if (!isset($cardRules[$cardType])) {
                return response()->json(['message' => 'Invalid card type selected.'], 422);
            }
    
            $existingCard = Card::where('user_id', $user->id)->where('card_type', $cardType)->first();
            if ($existingCard) {
                return response()->json(['message' => 'You already own this type of card.'], 403);
            }
    
            $totalDeposit = Transaction::where('user_id', $user->id)
                ->whereIn('type', ['Bank Deposit', 'Crypto Deposit'])
                ->sum('amount');
    
            if ($totalDeposit < $cardRules[$cardType]['required_tx']) {
                return response()->json(['message' => 'You need at least $' . $cardRules[$cardType]['required_tx'] . ' in total deposits to request this card.'], 403);
            }
    
            $cardNumber = rand(4000,4999).' '.rand(1000,9999).' '.rand(1000,9999).' '.rand(1000,9999);
            $month = str_pad(rand(1,12), 2, '0', STR_PAD_LEFT);
            $year = now()->addYears(3)->format('y');
            $ccv = rand(100,999);
    
            $card = Card::create([
                'user_id'   => $user->id,
                'fullname'  => $user->first_name . ' ' . $user->last_name,
                'cardnum'   => $cardNumber,
                'month'     => $month,
                'year'      => $year,
                'ccv'       => $ccv,
                'card_type' => $cardType,
                'type'      => 'Prepaid Card',
                'balance'   => $cardRules[$cardType]['limit'],
            ]);
    
            $maskedCardNumber = '**** **** **** ' . substr($card->cardnum, -4);
            $body = "Your new " . strtoupper(str_replace('_', ' ', $card->card_type)) . " has been created. Card Number: {$maskedCardNumber} Expiry: {$card->month}/{$card->year} CVV: {$card->ccv} Balance: $" . number_format($card->balance, 2);
    
            Mail::to($user->email)->send(new NewNotification($body, 'Your New Card Details', $user->first_name));
    
            return response()->json(['message' => 'Card request successful!']);
        } catch (\Exception $e) {
            Log::error('CreateCard Error: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to create card.'], 500);
        }
    }
    

    public function submitKYC(Request $request)
    {
        $user = Auth::user();
    
        try {
            $data = $request->validate([
                'first_name'     => 'required|string|max:100',
                'last_name'      => 'required|string|max:100',
                'email'          => 'required|email|max:150',
                'phone_number'   => 'required|string|max:20',
                'dob'            => 'required|date',
                'country'        => 'required|string|max:100',
                'document_type'  => 'required|string|in:National ID,Passport,Driver\'s License',
                'frontimg'       => 'required|image|mimes:jpeg,png,jpg|max:2048',
                'backimg'        => 'required|image|mimes:jpeg,png,jpg|max:2048',
            ]);
    
            $frontImagePath = $request->file('frontimg')->store('kyc', 'public');
            $backImagePath  = $request->file('backimg')->store('kyc', 'public');
    
            $kyc = KYC::updateOrCreate(
                ['user_id' => $user->id],
                [
                    'first_name'    => $data['first_name'],
                    'last_name'     => $data['last_name'],
                    'email'         => $data['email'],
                    'phone_number'  => $data['phone_number'],
                    'dob'           => $data['dob'],
                    'country'       => $data['country'],
                    'document_type' => $data['document_type'],
                    'frontimg'      => $frontImagePath,
                    'backimg'       => $backImagePath,
                    'status'        => 'Pending',
                ]
            );
    
            $user->update([
                'first_name'   => $data['first_name'],
                'last_name'    => $data['last_name'],
                'phone'        => $data['phone_number'],
                'country'      => $data['country'],
                'dob'          => $data['dob'],
            ]);
    
            return response()->json([
                'message'  => 'KYC Submitted Successfully! Awaiting Verification.',
                'redirect' => route('dashboard')
            ]);
    
        } catch (\Exception $e) {
            Log::error('KYC Submission Failed', ['error' => $e->getMessage()]);
            return response()->json(['message' => 'An error occurred while submitting KYC.'], 500);
        }
    }
    
    public function processBuyCrypto(Request $request)
    {
        try {
            $data = $request->validate([
                'payment_source'    => 'required|string',
                'amount'            => 'required|numeric|min:1',
                'wallet_address'    => 'required|string',
                'crypto'            => 'required|string',
            ]);
    
            $user = Auth::user();
    
            if (Str::startsWith($data['payment_source'], 'wallet_')) {
                $paymentMethod = 'wallet';
                $symbol = Str::after($data['payment_source'], 'wallet_');
    
                $wallet = UserWallet::where('user_id', $user->id)
                            ->whereHas('currency', function($q) use ($symbol) {
                                $q->where('symbol', $symbol);
                            })->firstOrFail();
    
                $balance = $wallet->balance;
    
            } elseif (Str::startsWith($data['payment_source'], 'card_')) {
                $paymentMethod = 'card';
    
                $cardId = Str::after($data['payment_source'], 'card_');
                $card = Card::where('id', $cardId)->where('user_id', $user->id)->firstOrFail();
    
                switch ($card->card_type) {
                    case 'visa_card':
                        $cardDisplayName = 'Visa Card';
                        break;
                    case 'master_card':
                        $cardDisplayName = 'Master Card';
                        break;
                    case 'gold_card':
                        $cardDisplayName = 'Gold Card';
                        break;
                    default:
                        $cardDisplayName = strtoupper(str_replace('_', ' ', $card->card_type));
                }
    
                $balance = $card->balance;
    
            } else {
                return response()->json(['message' => 'Invalid payment source.'], 422);
            }
    
            $cryptoSymbol = strtoupper($data['crypto']);
            $cryptoPrice = CryptoPrice::where('symbol', $cryptoSymbol)->value('price');
    
            if (!$cryptoPrice) {
                return response()->json(['message' => 'Crypto price unavailable.'], 422);
            }
    
            $cryptoAmount = $data['amount'] / $cryptoPrice;
            $fee = $data['amount'] * 0.015;
            $totalDebit = $data['amount'];
    
            if ($balance < $totalDebit) {
                return response()->json(['message' => 'Insufficient balance.'], 422);
            }
    
            if ($paymentMethod === 'wallet') {
                $wallet->balance -= $totalDebit;
                $wallet->save();
            } else {
                $card->balance -= $totalDebit;
                $card->save();
    
                CardTransaction::create([
                    'user_id'     => $user->id,
                    'title'       => 'Crypto Purchase',
                    'description' => "Purchase of {$cryptoAmount} {$cryptoSymbol}",
                    'date'        => now(),
                    'amount'      => -$totalDebit
                ]);
            }
    
            $ref = '#BUY-' . strtoupper(Str::random(10));
    
            $send = Transaction::create([
                'user_id'        => $user->id,
                'type'           => 'Buy Crypto',
                'from'           => $paymentMethod === 'wallet' ? 'USD Wallet' : 'Credit Card',
                'to'             => $data['wallet_address'],
                'currency_id'    => $paymentMethod === 'wallet' ? $wallet->currency_id : null,
                'amount'         => $data['amount'],
                'crypto_address' => $data['wallet_address'],
                'symbol'         => $cryptoSymbol,
                'payment_mode'   => $paymentMethod === 'card' ? 'Prepaid ' . $cardDisplayName : 'USD Wallet',
                'ref'            => $ref,
                'status'         => 'Pending',
            ]);
    
            $messageBody = 
            "Hello {$user->first_name},\n\n" .
            "You initiated a crypto purchase of {$cryptoAmount} {$cryptoSymbol}, " .
            "worth $" . number_format($data['amount'], 2) . ".\n" .
            "Payment Method: " . strtoupper($paymentMethod) . "\n" .
            "Destination Wallet: {$data['wallet_address']}\n\n" .
            "Reference: {$ref}\n" .
            "Status: Pending Processing.";
    
            Mail::to($user->email)->send(new NewNotification(
                $messageBody,
                'Crypto Purchase Initiated',
                $user->first_name
            ));
    
            return response()->json([
                'message'  => 'Your crypto purchase has been initiated. You will receive your coins shortly.',
                'redirect' => route('status', $send->id)
            ]);
        } catch (\Exception $e) {
            Log::error('ProcessBuyCrypto Failed', ['error' => $e->getMessage()]);
            return response()->json(['message' => 'An error occurred while processing your crypto purchase.'], 500);
        }
    }
    

    public function processBillPayment(Request $request)
    {
        $request->validate([
            'amount'            => 'required|numeric|min:1',
            'payment_source'    => 'required|string',
            'bill_id'           => 'required|exists:bills,id',
            'account_number'    => 'required|string',
            'customer_name'     => 'required|string',
        ]);
    
        $user = Auth::user();
    
        // Fetch Bill
        $bill = Bill::findOrFail($request->bill_id);
    
        // Check if payment is via Wallet or Card
        if ($request->payment_source !== 'card') {
            // Crypto Wallet Payment
            $wallet = UserWallet::with('currency')
                ->where('user_id', $user->id)
                ->whereHas('currency', function($q) use ($request) {
                    $q->where('symbol', $request->payment_source);
                })->first();
    
            if (!$wallet) {
                return back()->with('error', 'Selected wallet not found.');
            }
    
            $cryptoPrice = CryptoPrice::where('symbol', $request->payment_source)->first();
            if (!$cryptoPrice || $cryptoPrice->price <= 0) {
                return back()->with('error', 'Unable to fetch crypto price.');
            }
    
            // Convert USD to Crypto
            $cryptoAmount = round($request->amount / $cryptoPrice->price, 8);
    
            if ($wallet->balance < $cryptoAmount) {
                return back()->with('error', 'Insufficient wallet balance.');
            }
    
            // Deduct from wallet
            $wallet->balance -= $cryptoAmount;
            $wallet->save();
    
            $paymentMode = $wallet->currency->name . ' Wallet';
            $from = 'From My ' . $wallet->currency->name . ' Wallet';
    
        } else {
            // Card Payment
            $paymentMode = 'Card Payment';
            // Assuming card processing is external or balance check isn't required
        }
        $settings = Settings::find(1);
    
        // Log Transaction
        Transaction::create([
            'user_id'           => $user->id,
            'type'              => 'Bills Payment',
            'from'              => $from, 
            'to'                => $bill->name,
            'currency_id'       => isset($wallet) ? $wallet->currency->id : null,
            'amount'            => $request->amount,
            'fee'               => 0,
            'payment_mode'      => $paymentMode,
            'status'            => 'Completed',
            'transaction_time'  => now(),
            'ref'               => strtoupper(uniqid('BILL-')),
            'account_number'    => $request->account_number,
            'bank_account_name' => $request->customer_name,
            'symbol'            => $request->payment_source,
        ]);

        Mail::to($user->email)->send(new NewNotification(
            "Your payment for {$bill->name} of  {$settings->currency}{$request->amount} was successful.",
            'Bill Payment Confirmation',
            $user->first_name
        ));
    
        return response()->json([
            'success' => true,
            'message' => 'Bill payment successful.',
            'redirect' => route('dashboard')
        ]);
        
    }
    

    public function toggleSecuritySettings(Request $request)
    {
        $request->validate([
            'type'  => 'required|in:otp,twofa',
            'value' => 'required|in:yes,no',
        ]);
    
        $user = Auth::user();
    
        if ($request->type == 'otp') {
            $user->loginotp = $request->value;
        } elseif ($request->type == 'twofa') {
            $user->notify_status = $request->value;
        }
    
        $user->save();
    
        return response()->json([
            'success' => true,
            'message' => ucfirst($request->type) . ' setting updated to ' . strtoupper($request->value)
        ]);
    }
    

public function verify2FA(Request $request)
{
    $request->validate([
        'one_time_password' => 'required|digits:6',
    ]);

    $google2fa = new Google2FA();

    $user = Auth::user();
    $valid = $google2fa->verifyKey(decrypt($user->two_factor_secret), $request->one_time_password);

    if ($valid) {
        $user->notify_status = 'yes';  // Mark 2FA as enabled
        $user->save();

        return redirect()->route('dashboard')->with('success', '2FA Enabled Successfully!');
    }

    return back()->with('error', 'Invalid OTP, please try again.');
}


}