<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Settings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use App\Mail\WelcomeEmail;
use App\Models\Currency;
use App\Models\UserWallet;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;

class RegisterController extends Controller
{
    public function register(Request $request)
    {
        try {
            $settings = Settings::find(1);

            $rules = [
                'username' => ['required', 'string'],
                'email'    => ['required', 'string', 'email', 'unique:users'],
                'password' => [
                    'required', 'string', 'min:8',
                    'regex:/[A-Z]/',
                    'regex:/[0-9]/',
                    'regex:/[!@#$%^&*(),.?":{}|<>[\]]/'
                ],
            ];
            $rules['email'][] = $settings->captcha === 'true' ? 'max:55' : 'max:255';

            $validator = Validator::make($request->all(), $rules);
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $user = User::create([
                'username' => $request->username,
                'email'    => $request->email,
                'status'   => 'active',
                'password' => Hash::make($request->password),
            ]);

            $currencies = Currency::all();
            foreach ($currencies as $currency) {
                UserWallet::create([
                    'user_id'     => $user->id,
                    'currency_id' => $currency->id,
                    'balance'     => 0,
                ]);
            }

            event(new Registered($user));

            Mail::to($user->email)->send(new WelcomeEmail($user));

            Auth::login($user);

            return response()->json([
                'success'  => true,
                'redirect' => route('dashboard'),
            ], 201);

        } catch (\Exception $e) {
            Log::error('Registration Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred during registration.'
            ], 500);
        }
    }
}
