<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\UserWallet;
use App\Models\CryptoPrice;
use App\Models\Currency;
use App\Models\Card;
use App\Models\Transaction;
use Illuminate\Support\Str;

class TopupController extends Controller
{
    public function topup(Request $request)
    {
        $request->validate([
            'user_id'        => 'required|exists:users,id',
            'currency_symbol'=> 'required|string',
            'amount'         => 'required|numeric|min:1',
            't_type'         => 'required|in:Crypto Deposit,Bank Deposit,Crypto Withdrawal,Bank Withdrawal',
            'from'           => 'nullable|string',
        ]);

        $user = User::findOrFail($request->user_id);
        $symbol = $request->currency_symbol;
        $amountUSD = $request->amount;
        $type = $request->t_type;

        $ref = '#TOP-' . strtoupper(Str::random(10));

        // Handle Card Top-up (For Fiat Bank Deposits/Withdrawals via Card)
        if (in_array($symbol, ['visa_card', 'master_card', 'gold_card'])) {
            $card = Card::where('user_id', $user->id)
                        ->where('card_type', $symbol)
                        ->first();

            if (!$card) {
                return redirect()->back()->with('error', 'User does not own this card type.');
            }

            if (str_contains($type, 'Deposit')) {
                $card->balance += $amountUSD;
            } else {
                if ($card->balance < $amountUSD) {
                    return redirect()->back()->with('error', 'Insufficient card balance.');
                }
                $card->balance -= $amountUSD;
            }
            $card->save();

            Transaction::create([
                'user_id'     => $user->id,
                'type'        => $type,
                'from'        => 'Admin',
                'to'          => ucfirst(str_replace('_', ' ', $symbol)),
                'amount'      => $amountUSD,
                'symbol'      => strtoupper($symbol),
                'payment_mode'=> ucfirst(str_replace('_', ' ', $symbol)),
                'status'      => 'completed',
                'ref'         => $ref,
                'transaction_time' => now(),
            ]);

        } else {
            // Handle Crypto or Bank Wallet Top-up
            if ($symbol == 'USD') {
                // === BANK DEPOSIT / WITHDRAWAL ===
                if (str_contains($type, 'Deposit')) {
                    // Bank Deposit
                    Transaction::create([
                        'user_id'     => $user->id,
                        'type'        => $type,
                        'from'        => 'Admin',
                        'to'          => 'Bank Account',
                        'amount'      => $amountUSD,
                        'symbol'      => 'USD',
                        'payment_mode'=> 'Bank Transfer',
                        'status'      => 'completed',
                        'ref'         => $ref,
                        'transaction_time' => now(),
                    ]);
                } else {
                    // Bank Withdrawal
                    Transaction::create([
                        'user_id'     => $user->id,
                        'type'        => $type,
                        'from'        => 'Bank Account',
                        'to'          => 'Admin',
                        'amount'      => $amountUSD,
                        'symbol'      => 'USD',
                        'payment_mode'=> 'Bank Transfer',
                        'status'      => 'completed',
                        'ref'         => $ref,
                        'transaction_time' => now(),
                    ]);
                }

            } else {
                // === CRYPTO DEPOSIT / WITHDRAWAL ===
                $currency = Currency::where('symbol', $symbol)->first();
                if (!$currency) {
                    return redirect()->back()->with('error', 'Invalid cryptocurrency selected.');
                }

                $wallet = UserWallet::where('user_id', $user->id)
                                    ->where('currency_id', $currency->id)
                                    ->first();

                if (!$wallet) {
                    return redirect()->back()->with('error', 'User wallet for selected currency not found.');
                }

                $cryptoPrice = CryptoPrice::where('symbol', $symbol)->value('price');
                if (!$cryptoPrice || $cryptoPrice == 0) {
                    return redirect()->back()->with('error', 'Crypto price unavailable.');
                }

                $cryptoAmount = $amountUSD / $cryptoPrice;

                if (str_contains($type, 'Deposit')) {
                    $wallet->balance += $cryptoAmount;
                } else {
                    if ($wallet->balance < $cryptoAmount) {
                        return redirect()->back()->with('error', 'Insufficient crypto balance.');
                    }
                    $wallet->balance -= $cryptoAmount;
                }
                $wallet->save();

                Transaction::create([
                    'user_id'     => $user->id,
                    'type'        => $type,
                    'from'        => 'Admin',
                    'to'          => $symbol . ' Wallet',
                    'currency_id' => $currency->id,
                    'amount'      => $amountUSD,
                    'symbol'      => $symbol,
                    'payment_mode'=> $symbol,
                    'status'      => 'completed',
                    'ref'         => $ref,
                    'transaction_time' => now(),
                ]);
            }
        }

        return redirect()->back()->with('success', $type . ' completed successfully!');
    }
}

